/*:
 * @target MZ
 * @plugindesc v1.5 ピクチャ上ホバーでツールチップ（Append追記＋再描画Refresh対応）
 * @author HS
 *
 * @command AddTooltip
 * @text Add Tooltip
 * @desc ピクチャにツールチップを設定（上書き）
 * @arg pictureId @type number @min 1
 * @arg text @type string
 * @arg offsetX @type number @min -999 @default 16
 * @arg offsetY @type number @min -999 @default -24
 * @arg anchor
 * @type select
 * @option cursor
 * @option topLeft
 * @option top
 * @option topRight
 * @option left
 * @option center
 * @option right
 * @option bottomLeft
 * @option bottom
 * @option bottomRight
 * @default cursor
 *
 * @command AppendTooltip
 * @text Append Tooltip
 * @desc ツールチップ文字列に追記（既存は保持）
 * @arg pictureId @type number @min 1
 * @arg text @type string
 * @arg position
 * @type select
 * @option append
 * @option prepend
 * @default append
 * @arg separator
 * @type string
 * @default \n
 *
 * @command ClearTooltipText
 * @text Clear Tooltip Text
 * @desc ツールチップの文字だけ消去（設定は保持）
 * @arg pictureId @type number @min 1
 *
 * @command RemoveTooltip
 * @text Remove Tooltip
 * @desc ツールチップ設定を削除
 * @arg pictureId @type number @min 1
 *
 * @command RefreshTooltip
 * @text Refresh Tooltip (Redraw)
 * @desc 内容は変更せず再描画のみ実行
 * @arg mode
 * @type select
 * @option window
 * @option picture
 * @default window
 * @arg pictureId
 * @type number
 * @min 1
 * @default 1
 * @desc mode=picture のとき対象ピクチャID
 * @arg keepPosition
 * @type boolean
 * @default true
 * @desc mode=window のとき現在位置を保ったまま再描画
 *
 * @help
 * ピクチャにカーソルを重ねた時、文字のツールチップを表示します。
 * v1.4で Append/Clear を追加、v1.5で Refresh（再描画のみ）を追加。
 *
 * ■使い方
 * 1) AddTooltip で初期文字を設定
 * 2) AppendTooltip を任意回数呼んで末尾/先頭に追記
 * 3) 表示を即反映したいとき RefreshTooltip を呼ぶ
 *    - mode=window … 今出ているウィンドウを再描画（keepPositionで位置維持）
 *    - mode=picture … 指定ピクチャIDのアンカー基準で位置再計算→再描画
 */

(() => {
  const NAME = "MEG_PictureTooltipMZ";
  // id -> {text, ox, oy, anchor}
  const DB = new Map();

  const toNum = (v, def) => {
    const n = Number(v ?? def);
    return Number.isFinite(n) ? n : def;
  };

  // ---------------- Plugin Commands ----------------
  PluginManager.registerCommand(NAME, "AddTooltip", a => {
    const id  = toNum(a.pictureId, 0);
    if (id <= 0) return;
    const txt = String(a.text ?? "");
    const ox  = toNum(a.offsetX, 16);
    const oy  = toNum(a.offsetY, -24);
    const anchor = (a.anchor ?? "cursor");
    DB.set(id, { text: txt, ox, oy, anchor });
  });

  PluginManager.registerCommand(NAME, "AppendTooltip", a => {
    const id  = toNum(a.pictureId, 0);
    if (id <= 0) return;
    const add = String(a.text ?? "");
    const position = (a.position ?? "append");
    let sep = String(a.separator ?? "\n");
    sep = sep.replace(/\\n/g, "\n").replace(/\\t/g, "\t").replace(/\\r/g, "\r");

    const cur = DB.get(id);
    if (!cur) { DB.set(id, { text: add, ox:16, oy:-24, anchor:"cursor" }); return; }
    const base = String(cur.text ?? "");
    const needsSep = base.length > 0 && add.length > 0;
    if (position === "prepend") cur.text = add + (needsSep ? sep : "") + base;
    else cur.text = base + (needsSep ? sep : "") + add;
  });

  PluginManager.registerCommand(NAME, "ClearTooltipText", a => {
    const id  = toNum(a.pictureId, 0);
    if (id <= 0) return;
    const cur = DB.get(id);
    if (cur) cur.text = "";
  });

  PluginManager.registerCommand(NAME, "RemoveTooltip", a => {
    DB.delete(toNum(a.pictureId, 0));
  });

  PluginManager.registerCommand(NAME, "RefreshTooltip", a => {
    const mode = (a.mode ?? "window");
    const keep = String(a.keepPosition ?? "true") === "true";
    if (mode === "window") {
      const scene = SceneManager._scene;
      scene?._refreshTooltipWindow(keep);
    } else {
      const id = toNum(a.pictureId, 0);
      if (id <= 0) return;
      const scene = SceneManager._scene;
      scene?._refreshTooltipByPicture(id);
    }
  });

  // ---------------- Tooltip Window ----------------
  class Window_HoverTooltip extends Window_Base {
    initialize() {
      super.initialize(new Rectangle(0,0,1,1));
      this.openness = 0;
      this._hold = 0;
      this._lastText = "";
      this._lastPictureId = 0;
      this.backOpacity = 192;
      this.contents.outlineWidth = 3;
    }

    // 複数行のサイズを計算（※元のまま）
    _measureMultiline(text) {
      const lines = String(text ?? "").split(/\r?\n/);
      let w = 0;
      this.resetFontSettings();
      for (const l of lines) {
        w = Math.max(w, Math.ceil(this.textSizeEx(l).width));
      }
      const h = this.fittingHeight(Math.max(1, lines.length));
      return { w, h };
    }

    showAt(pictureId, text, x, y) {
      this._lastPictureId = pictureId || 0;
      this._lastText = String(text ?? "");
      const { w, h } = this._measureMultiline(this._lastText);
      const needW = Math.min(w + this.padding * 2 + 12, Graphics.width);
      const needH = Math.min(h, Graphics.height);
      if (this.width !== needW || this.height !== needH) {
        this.move(this.x, this.y, needW, needH);
        this.createContents();
      }
      this._redrawText(this._lastText);
      // 画面内にクランプ
      this.x = Math.max(0, Math.min(x, Graphics.width  - this.width));
      this.y = Math.max(0, Math.min(y, Graphics.height - this.height));
      this._hold = 10;
      this.open();
    }

    // 現在位置を保ったまま、または再クランプして再描画
    refreshTextFromDB(text, keepPosition = true) {
      this._lastText = String(text ?? "");
      const px = this.x, py = this.y;
      const { w, h } = this._measureMultiline(this._lastText);
      const needW = Math.min(w + this.padding * 2 + 12, Graphics.width);
      const needH = Math.min(h, Graphics.height);
      if (this.width !== needW || this.height !== needH) {
        this.move(keepPosition ? px : this.x, keepPosition ? py : this.y, needW, needH);
        this.createContents();
      }
      this._redrawText(this._lastText);
      if (!keepPosition) {
        this.x = Math.max(0, Math.min(this.x, Graphics.width  - this.width));
        this.y = Math.max(0, Math.min(this.y, Graphics.height - this.height));
      }
      this._hold = 10;
      this.open();
    }

    _redrawText(text) {
      this.contents.clear();
      this.resetFontSettings();
      this.contents.fontBold = true;
      this.drawTextEx(String(text ?? ""), 0, 0);
    }

    update() {
      super.update();
      if (this._hold > 0) this._hold--;
      else this.close();
    }
  }

  // Scene_Map 拡張
  const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
  Scene_Map.prototype.createAllWindows = function() {
    _Scene_Map_createAllWindows.call(this);
    this._hoverTooltip = new Window_HoverTooltip();
    this.addWindow(this._hoverTooltip);
  };

  // 既存表示の再描画（windowモード）
  Scene_Map.prototype._refreshTooltipWindow = function(keepPosition) {
    const w = this._hoverTooltip;
    if (!w) return;
    // 最後に表示したピクチャの最新テキストで再描画
    const id = w._lastPictureId;
    const text = id > 0 && DB.get(id) ? DB.get(id).text : w._lastText;
    w.refreshTextFromDB(text, keepPosition);
  };

  // ピクチャID基準で再描画（pictureモード）
  Scene_Map.prototype._refreshTooltipByPicture = function(pictureId) {
    const cfg = DB.get(pictureId);
    if (!cfg) return;
    const sp = findPictureSprite(pictureId);
    if (!sp) return;
    const rc = spriteRect(sp); if (!rc) return;
    let base;
    if (cfg.anchor === "cursor") base = { x: TouchInput.x, y: TouchInput.y };
    else base = anchorPoint(rc, cfg.anchor) ?? { x: TouchInput.x, y: TouchInput.y };
    this._showTooltip(pictureId, cfg.text, base.x + cfg.ox, base.y + cfg.oy);
  };

  // 内部用：表示
  Scene_Map.prototype._showTooltip = function(pictureId, text, x, y) {
    this._hoverTooltip?.showAt(pictureId, text, x, y);
  };

  // ---------------- Helpers ----------------
  function spriteRect(s) {
    if (!s || !s.bitmap || !s.bitmap.isReady()) return null;
    const w = s.bitmap.width * Math.abs(s.scale.x);
    const h = s.bitmap.height * Math.abs(s.scale.y);
    const l = s.x - w * s.anchor.x;
    const t = s.y - h * s.anchor.y;
    return { l, t, r: l + w, b: t + h, w, h, cx: l + w/2, cy: t + h/2 };
  }

  function anchorPoint(rect, name) {
    switch (name) {
      case "topLeft":     return { x: rect.l,  y: rect.t  };
      case "top":         return { x: rect.cx, y: rect.t  };
      case "topRight":    return { x: rect.r,  y: rect.t  };
      case "left":        return { x: rect.l,  y: rect.cy };
      case "center":      return { x: rect.cx, y: rect.cy };
      case "right":       return { x: rect.r,  y: rect.cy };
      case "bottomLeft":  return { x: rect.l,  y: rect.b  };
      case "bottom":      return { x: rect.cx, y: rect.b  };
      case "bottomRight": return { x: rect.r,  y: rect.b  };
      default:            return null; // cursor
    }
  }

  function findPictureSprite(id) {
    const scene = SceneManager._scene;
    const sprset = scene?._spriteset;
    const cont = sprset?._pictureContainer;
    if (!cont) return null;
    for (const c of cont.children) {
      if (c && c._pictureId === id) return c;
    }
    return null;
  }

  // ---------------- Hover Detection ----------------
  const _Sprite_Picture_update = Sprite_Picture.prototype.update;
  Sprite_Picture.prototype.update = function() {
    _Sprite_Picture_update.call(this);

    // ▼追加：バックログ表示中はツールチップ無効化
    if (window.HSBL && typeof window.HSBL.isOpen === "function" && window.HSBL.isOpen()) {
      const scene = SceneManager._scene;
      if (scene && scene._hoverTooltip) {
        scene._hoverTooltip._hold = 0;
        scene._hoverTooltip.close();
      }
      return;
    }
    // ▲ここまで追加

    const cfg = DB.get(this._pictureId);
    if (!cfg || !this.visible || this.opacity === 0) return;
    const rc = spriteRect(this); if (!rc) return;

    const mx = TouchInput.x, my = TouchInput.y;
    if (mx >= rc.l && mx <= rc.r && my >= rc.t && my <= rc.b) {
      let base;
      if (cfg.anchor === "cursor") base = { x: mx, y: my };
      else base = anchorPoint(rc, cfg.anchor) ?? { x: mx, y: my };
      SceneManager._scene?._showTooltip(this._pictureId, cfg.text, base.x + cfg.ox, base.y + cfg.oy);
    }
  };
})();

